within ;
package TankWithController

  package Interfaces

    connector ReadSignal "Reading fluid level"
      Real val(unit = "m");
      annotation (Icon(graphics={Rectangle(
              extent={{-100,100},{100,-100}},
              lineColor={28,108,200},
              fillColor={0,128,255},
              fillPattern=FillPattern.Solid)}));
    end ReadSignal;

    connector ActSignal "Signal to actuator for setting valve position"
      Real act;
      annotation (Icon(graphics={Rectangle(
              extent={{-100,100},{100,-100}},
              lineColor={28,108,200},
              fillColor={255,128,0},
              fillPattern=FillPattern.Solid)}));
    end ActSignal;

    connector LiquidFlow "Liquid flow at inlets or outlets"
      Real lflow(unit = "m3/s");
      annotation (
        Icon(graphics={  Ellipse(extent = {{-100, 100}, {100, -100}}, endAngle = 360,
              lineColor={0,0,0},
              fillColor={85,255,85},
              fillPattern=FillPattern.Solid)}));
    end LiquidFlow;
  end Interfaces;

  package Functions

    function LimitValue
      input Real minV;
      input Real maxV;
      input Real inV;
      output Real outV;
    algorithm
      if inV < minV then
        outV := minV;
      elseif maxV < inV then
        outV := maxV;
      else
        outV := inV;
      end if;
    end LimitValue;
  end Functions;

  package Components

    model Tank
      Interfaces.ReadSignal tSensor
        "Connector, sensor reading tank level (m)" annotation (Placement(
          visible=true,
          transformation(
            origin={-110,-50},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={-110,-50},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Interfaces.ActSignal tActuator
        "Connector, actuator controlling input flow" annotation (Placement(
          visible=true,
          transformation(
            origin={110,-50},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={110,-50},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Interfaces.LiquidFlow qIn
        "Connector, flow (m3/s) through input valve" annotation (Placement(
          visible=true,
          transformation(
            origin={-110,50},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={-110,50},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Interfaces.LiquidFlow qOut
        "Connector, flow (m3/s) through output valve" annotation (Placement(
          visible=true,
          transformation(
            origin={110,50},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={110,50},
            extent={{-10,-10},{10,10}},
            rotation=0)));
    equation

    annotation(
        Diagram(graphics = {Text(origin = {4, 4}, extent = {{-60, 44}, {60, -44}}, textString = "Tank")}),
        Icon);end Tank;

    model LiquidSource
      Interfaces.LiquidFlow qOut annotation (Placement(
          visible=true,
          transformation(
            origin={110,0},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={110,0},
            extent={{-10,-10},{10,10}},
            rotation=0)));
    end LiquidSource;

    partial model BaseController
    parameter Real Ts(unit = "s") = 0.1 "Ts - Time period between discrete samples  discrete sampled";
      parameter Real K = 2 "Gain";
      parameter Real T(unit = "s") = 10 "Time constant - continuous";
      parameter Real ref "Reference level";
      Real error "Deviation from reference level";
      Real outCtr "Output control signal";
      Interfaces.ReadSignal cIn
        "Input sensor level,  connector" annotation (Placement(
          visible=true,
          transformation(
            origin={-110,0},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={-110,0},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Interfaces.ActSignal cOut
        "Control to actuator, connector" annotation (Placement(
          visible=true,
          transformation(
            origin={110,0},
            extent={{-10,-10},{10,10}},
            rotation=0),
          iconTransformation(
            origin={110,0},
            extent={{-10,-10},{10,10}},
            rotation=0)));
    equation
      error = ref - cIn.val;
      cOut.act = outCtr;
    end BaseController;

    model PIcontinuousController
    extends Components.BaseController;
    end PIcontinuousController;
  end Components;

  package Examples

    model FlatTankWithController
      // Tank related variables and parameters
      parameter Real flowLevel(unit="m3/s")=0.02;
      parameter Real area(unit="m2") =      1;
      parameter Real flowGain(unit="m2/s")= 0.05;
      Real           h(start=0,unit="m")   "Tank level";
      Real           qInflow(unit="m3/s")  "Flow through input valve";
      Real           qOutflow(unit="m3/s") "Flow through output valve";
      // Controller related variables and parameters
      parameter Real K=2                   "Gain";
      parameter Real T(unit="s")= 10       "Time constant";
      parameter Real minV=0, maxV=10;    // Limits for flow output
      Real           ref = 0.25  "Reference level for control";
      Real           error       "Deviation from reference level";
      Real           outCtr      "Control signal without limiter";
      Real           x           "State variable for controller";
    equation
      assert(minV>=0,"minV must be greater or equal to zero");//
      der(h) = (qInflow-qOutflow)/area;   // Mass balance equation
      qInflow  = if time>150 then 3*flowLevel else flowLevel;
      qOutflow =Functions.LimitValue(
            minV,
            maxV,
            -flowGain*outCtr);
      error  = ref-h;
      der(x) = error/T;
      outCtr = K*(error+x);
    end FlatTankWithController;

    model TankWithController
      Components.PIcontinuousController controller(ref=0.25)
        annotation (Placement(visible=true, transformation(
            origin={10,-30},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Components.Tank tank(area=1) annotation (Placement(
            visible=true, transformation(
            origin={10,10},
            extent={{-10,-10},{10,10}},
            rotation=0)));
      Components.LiquidSource source(flowLevel=0.02)
        annotation (Placement(visible=true, transformation(
            origin={-30,30},
            extent={{-10,-10},{10,10}},
            rotation=0)));
    equation
      connect(tank.tSensor, controller.cIn) annotation (Line(points={{-1,5},{
              -20,5},{-20,-30},{-1,-30}}, color={28,108,200}));
      connect(controller.cOut, tank.tActuator) annotation (Line(points={{21,-30},
              {40,-30},{40,5},{21,5}}, color={28,108,200}));
      connect(source.qOut, tank.qIn) annotation (Line(points={{-19,30},{-10,30},
              {-10,15},{-1,15}}, color={0,0,0}));
    end TankWithController;
  end Examples;
end TankWithController;